/*
 *
 *  Copyright (c) 2008-2011 Erich Hoover
 *
 *  elfres - Adds resources into ELF files
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * To provide feedback, report bugs, or otherwise contact me:
 * ehoover at mines dot edu
 *
 */

#if GUI_ENABLED

/* Handle GLADE files */
#include <gtk/gtk.h>
#include <glade/glade.h>

/* Use libr GTK+ convenience routines */
#include <libr-gtk.h>

/* Internationalization support */
#include <libr-i18n.h>

#include "elfres.h"

/* Global GUI Resources */
GladeXML *glade = NULL;
GtkBuilder *builder = NULL;
libr_file *gui_handle = NULL;

/*
 * Return the requested widget depending on the resource method
 */
GtkWidget *get_widget(const char *name)
{
	if(glade)
		return (GtkWidget *) glade_xml_get_widget(glade, name);
	if(builder)
		return (GtkWidget *) gtk_builder_get_object(builder, name);
	return NULL;
}

/*
 * Open a gui error dialog for problems
 * NOTE: call errorf() and it will choose to use this function if appropriate.
 */ 
void gui_err(char *format, ...)
{
	GtkLabel *errmsg = GTK_LABEL(get_widget("errmsg"));
	GtkWidget *errdialog = get_widget("errdialog");
	char buffer[ERROR_BUF];
	va_list args;
	
	va_start(args, format);
	vsnprintf(buffer, ERROR_BUF, format, args);
	va_end(args);
	gtk_label_set_text(errmsg, buffer);
	gtk_widget_show(errdialog);
}

/*
 * Allow the user to choose a resource to add to the ELF binary
 */
GUI_FN void gui_add_resource(GtkFileChooser *dialog, void *_m)
{
	GtkTreeView *resourcelist = GTK_TREE_VIEW(get_widget("resourcelist"));
	GtkListStore *lstore = GTK_LIST_STORE(gtk_tree_view_get_model(resourcelist));
	char *filename = gtk_file_chooser_get_filename(dialog);
	char *abbrvname = strdup(strrchr(filename, '/')+1);
	GtkTreeIter iter;
	
	/* Add the file to the binary */
	if(!add_resource(gui_handle, abbrvname, filename))
		return; /* Failed to add file */

	/* Add the file to the GUI list */
	gtk_list_store_append(lstore, &iter);
	gtk_list_store_set(lstore, &iter, 0, abbrvname, -1);
}

/*
 * Allow the user to extract a resource from an ELF binary
 */
GUI_FN void gui_save_resource(GtkFileChooser *dialog, void *_m)
{
	GtkTreeView *resourcelist = GTK_TREE_VIEW(get_widget("resourcelist"));
	GtkTreeModel *lstore = GTK_TREE_MODEL(gtk_tree_view_get_model(resourcelist));
	GtkTreeSelection *selection = gtk_tree_view_get_selection(resourcelist);
	char *filename = gtk_file_chooser_get_filename(dialog);
	char *abbrvname = strdup(strrchr(filename, '/')+1);
	char *resource = NULL;
	GtkTreeIter iter;
	
	if(!gtk_tree_selection_get_selected(selection, &lstore, NULL))
	{
		errorf(_("No resource selected."));
		return;
	}
	gtk_tree_selection_get_selected(selection, &lstore, &iter);
	gtk_tree_model_get(lstore, &iter, 0, &resource, -1);
	
	/* Save the file */
	get_resource(gui_handle, resource, filename);
}

/*
 * Remove the selected resource from the ELF binary
 */
GUI_FN void gui_remove_resource(GtkWidget *widget, void *_m)
{
	GtkTreeView *resourcelist = GTK_TREE_VIEW(get_widget("resourcelist"));
	GtkTreeModel *lstore = GTK_TREE_MODEL(gtk_tree_view_get_model(resourcelist));
	GtkTreeSelection *selection = gtk_tree_view_get_selection(resourcelist);
	char *resource = NULL;
	GtkTreeIter iter;
	
	if(!gtk_tree_selection_get_selected(selection, &lstore, NULL))
	{
		errorf(_("No resource selected."));
		return;
	}
	gtk_tree_selection_get_selected(selection, &lstore, &iter);
	gtk_tree_model_get(lstore, &iter, 0, &resource, -1);
	if(!libr_clear(gui_handle, resource))
	{
		errorf(_("failed to remove resource: %s"), libr_errmsg());
		return;
	}
	gtk_list_store_remove(GTK_LIST_STORE(lstore), &iter);
}

/*
 * Re-save the ELF binary with the requested changes
 */
GUI_FN void gui_apply_changes(GtkWidget *widget, void *_m)
{
	GtkFileChooser *dialog = GTK_FILE_CHOOSER(get_widget("openfile"));
	char *filename = gtk_file_chooser_get_filename(dialog);
	
	/* Write changes to disk */
	libr_close(gui_handle);
	/* Re-open handle */
	if((gui_handle = libr_open(filename, LIBR_READ_WRITE)) == NULL)
	{
		errorf(_("failed to re-open the executable (%s) for resources: %s"), filename, libr_errmsg());
		return;
	}
}

/*
 * A row was selected, enable the save and remove options
 */
GUI_FN void gui_row_selected(void *_m)
{
	GtkWidget *remove = get_widget("remove");
	GtkWidget *save = get_widget("save");

	gtk_widget_set_sensitive(save, TRUE);
	gtk_widget_set_sensitive(remove, TRUE);
}

/*
 * A row was selected, enable the save and remove options
 */
GUI_FN void gui_open_save_dialog(GtkFileChooser *dialog, void *_m)
{
	GtkTreeView *resourcelist = GTK_TREE_VIEW(get_widget("resourcelist"));
	GtkTreeModel *lstore = GTK_TREE_MODEL(gtk_tree_view_get_model(resourcelist));
	GtkTreeSelection *selection = gtk_tree_view_get_selection(resourcelist);
	char *resource = NULL;
	GtkTreeIter iter;
	
	if(!gtk_tree_selection_get_selected(selection, &lstore, NULL))
	{
		errorf(_("No resource selected."));
		return;
	}
	gtk_tree_selection_get_selected(selection, &lstore, &iter);
	gtk_tree_model_get(lstore, &iter, 0, &resource, -1);
	gtk_file_chooser_set_filename(dialog, resource);
	gtk_widget_show(GTK_WIDGET(dialog));
}

/*
 * Allow the user to choose an ELF binary to edit
 */
GUI_FN void gui_open_file(GtkFileChooser *dialog, void *_m)
{
	GtkTreeView *resourcelist = GTK_TREE_VIEW(get_widget("resourcelist"));
	GtkLabel *activefile = GTK_LABEL(get_widget("activefile"));
	GtkCellRenderer *renderer = gtk_cell_renderer_text_new();
	char *filename = gtk_file_chooser_get_filename(dialog);
	char *abbrvname = strdup(strrchr(filename, '/')+1);
	GtkWidget *getfile = get_widget("getfile");
	GtkWidget *apply = get_widget("apply");
	GtkWidget *add = get_widget("add");
	GtkListStore *lstore;
	int i, res;
	
	if((gui_handle = libr_open(filename, LIBR_READ_WRITE)) == NULL)
	{
		errorf(_("failed to open the executable (%s) for resources: %s"), filename, libr_errmsg());
		return;
	}
	res = libr_resources(gui_handle);
	
	lstore = gtk_list_store_new(1, G_TYPE_STRING);
	for(i=0;i<res;i++)
	{
		char *name = libr_list(gui_handle, i);
		GtkTreeIter iter;
		
		if(name == NULL)
		{
			printf("error!\n");
			continue;
		}
		gtk_list_store_append(lstore, &iter);
		gtk_list_store_set(lstore, &iter, 0, name, -1);
		free(name);
	}
	gtk_tree_view_insert_column_with_attributes(resourcelist, -1, "Resource", renderer, "text", 0, NULL);
	gtk_tree_view_set_model(resourcelist, GTK_TREE_MODEL(lstore));
	
	gtk_label_set_text(activefile, abbrvname);
	gtk_widget_set_sensitive(add, TRUE);
	gtk_widget_set_sensitive(apply, TRUE);
	gtk_widget_set_sensitive(getfile, FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(resourcelist), TRUE);
}

/*
 * Started from desktop
 */
int main_gui(int argc, char **argv)
{
	GtkWindow *window = NULL;
	GList *icons = NULL;
	
	/* Initialize GTK and use libr to automatically load the user interface */
	gtk_init(&argc, &argv);
	/* Example of using GtkBuilder user interfaces */
	if(!libr_gtk_autoload(&builder, &icons, TRUE))
	{
		printf("failed to load GtkBuilder file, trying glade...\n");
		/* Example of using libglade user interfaces */
		if(!libr_glade_autoload(&glade, &icons, TRUE))
		{
			errorf("failed to obtain glade resource: %s", libr_errmsg());
			return EX_SOFTWARE;
		}
		window = GTK_WINDOW(glade_xml_get_widget(glade, "elfres"));
		glade_xml_signal_autoconnect(glade);
	}
	else
	{
		window = GTK_WINDOW(gtk_builder_get_object(builder, "elfres"));
		gtk_builder_connect_signals(builder, NULL);
	}
	/*
	 * Example of setting an icon for a window.
	 *
	 * NOTE: libr_gtk_autoload(..., ..., TRUE) automatically sets the default
	 * window icon to the icon of the executable, this is for demonstrative
	 * purposes in case you have a multi-icon application.
	 */
	if(icons != NULL)
		gtk_window_set_icon_list(window, icons);
	/* Wait for and respond to user operations */
	gui_running = TRUE;
	gtk_main();
	/* If a file was opened then cleanup the libr handle */
	if(gui_handle)
		libr_close(gui_handle);
	return EX_OK;
}

#endif /* GUI_ENABLED */
